package gov.va.med.mhv.usermgmt.persist.ldap;

import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import junit.framework.TestCase;
import netscape.ldap.LDAPAttribute;
import netscape.ldap.LDAPAttributeSet;
import netscape.ldap.LDAPConnection;
import netscape.ldap.LDAPEntry;
import netscape.ldap.LDAPException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;

/**
 * Abstract test case that provides convenience methods for other Ldap test cases.
 *
 * @author Jon Crater Apr 21, 2006 1:51:02 PM
 */
public class AbstractLdapTests extends TestCase {
	protected static final String USER_PREFIX = "testUser-";
	protected static final String PASSWORD = "password";
	protected static final int NO_SUCH_OBJECT = LDAPException.NO_SUCH_OBJECT;
	protected final Log logger = LogFactory.getLog(getClass());

	protected LDAPConnection conn;
	protected LdapEnvironment env;
	protected LDAPEntry testUserEntry;
	protected String testUserDn;
	protected String testUserName;

	protected void setUp() throws Exception {
		logger.info("==> setUp " + getName() + " <==");

		initializeEnvironment();
		connect();

		testUserEntry = createUserEntryForTest();
		uidForEntry(testUserEntry);

		onSetup();
	}

	protected void tearDown() throws Exception {
		logger.info("==> tearDown " + getName() + " <==");

		try {
			deleteEntry(testUserDn);
		}
		catch(LDAPException e) {
			// don't fail if we can't find the entry to delete.  some tests delete during a normal test run
			if(e.getLDAPResultCode() != NO_SUCH_OBJECT)
				throw e;
		}
		conn.disconnect();

		onTearDown();
	}

	protected LDAPEntry readEntry(String dn) throws LDAPException {
		logger.debug("Reading entry with dn '" + dn + "'");
		return conn.read(dn);
	}

	protected void deleteEntry(String dn) throws LDAPException {
		logger.debug("Deleting entry with dn '" + dn + "'");
		conn.delete(dn);
	}

	protected LDAPEntry createUserEntryForTest() throws Exception {
		testUserName = getTestUserName();
		testUserDn = LdapUtil.buildUserDn(testUserName);

		LdapUpdate update = new LdapUpdate(LdapUtil.buildUserDn(testUserName));
		Map values = new HashMap(5);
		values.put(env.getObjectClassAttribute(), env.getObjectClassValues());
		values.put(env.getUserNameAttribute(), testUserName);
		values.put(env.getUserPasswordAttribute(), getTestUserPassword());

		// until and unless we pass first name and last name, these attributes have to be set
		// to comply with the schema rules.  for now, set them both to the userName
		values.put(env.getCommonNameAttribute(), testUserName);
		values.put(env.getSurNameAttribute(), testUserName);

		// persist it
		update.addCommand(LdapEntryModification.add(values));
		update.executeUpdate();

		logger.debug("Created test user at dn '" + testUserDn + "'");
		return readEntry(testUserDn);
	}

	protected String uidForEntry(LDAPEntry entry) {
		LDAPAttributeSet attSet = entry.getAttributeSet();
		LDAPAttribute att = attSet.getAttribute(env.getUserNameAttribute());
		String[] values = att.getStringValueArray();
		assertNotNull(values);
		assertTrue(values.length == 1);
		return values[0];
	}

	protected void initializeEnvironment() throws IOException {
		ResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
		Resource [] resources = resolver.getResources("classpath*:**/ldap.properties");
		
		String fName = resources[0].getFile().getParent();
		
		System.setProperty("config.location", fName);
		env = LdapEnvironment.getInstance();
	}

	protected void connect() throws LDAPException {
		conn = new LDAPConnection();
		conn.connect(env.getHost(), env.getPort(), env.getAdminPrincipal(), env.getAdminPassword());
	}

	protected void onSetup() throws Exception {
	}

	protected void onTearDown() throws Exception {
	}

	protected String getTestUserName() {
		return USER_PREFIX + String.valueOf(System.currentTimeMillis());
	}

	protected String getTestUserPassword() {
		return PASSWORD;
	}
}
